<?php

namespace App\Middleware;

use App\Interfaces\CaptchaVerifierInterface;
use Psr\Http\Message\ServerRequestInterface as Request;
use Psr\Http\Server\RequestHandlerInterface as RequestHandler;
use Slim\Psr7\Response;
use App\Services\RecaptchaVerifier;
use App\Services\TurnstileVerifier;

class CaptchaMiddleware
{
    private float $minimumScore = 0.5;

    public function __invoke(Request $request, RequestHandler $handler): Response
    {
        $config = requireBackendFile($request, 'config.php');
        $spamConfig = $config['spam'] ?? [];

        $verifier = isset($spamConfig['service']) ? match ($spamConfig['service']) {
            'turnstile' => new TurnstileVerifier(
                $spamConfig['turnstile']['secret_key'],
                $spamConfig['turnstile']['site_key']
            ),
            'recaptcha' => new RecaptchaVerifier(
                $spamConfig['recaptcha']['secret_key'],
                $spamConfig['recaptcha']['site_key']
            ),
            'none' => null
        } : null;

        // Skip verification if no verifier is set
        if (!$verifier) {
            return $handler->handle($request);
        }

        $token = $request->getParsedBody()['captcha_token'] ?? null;

        if (!$token) {
            return $this->error('No captcha token provided');
        }

        $verification = $verifier->verify($token);

        if (!$verification['success']) {
            return $this->error('Captcha verification failed');
        }

        if (isset($verification['score']) && $verification['score'] < $this->minimumScore) {
            return $this->error('Captcha score too low');
        }

        return $handler->handle($request);
    }

    private function error(string $message): Response
    {
        $response = new Response();
        $response->getBody()->write(json_encode([
            'error' => $message
        ]));
        return $response->withStatus(400)->withHeader('Content-Type', 'application/json');
    }
} 