<?php

namespace App\Middleware;

use Psr\Http\Message\ServerRequestInterface as Request;
use Psr\Http\Server\RequestHandlerInterface as RequestHandler;
use Psr\Http\Message\ResponseInterface;
use Psr\Container\ContainerInterface;
use InvalidArgumentException;
use Psr\Log\LoggerInterface;

class ComponentValidationMiddleware
{
    public function __construct(
        private LoggerInterface $logger,
        private ContainerInterface $container
    ) {}

    public function __invoke(Request $request, RequestHandler $handler): ResponseInterface
    {
        // Get data from either POST or GET
        $data = $request->getMethod() === 'POST' 
            ? $request->getParsedBody() 
            : $request->getQueryParams();

        $this->validateDataArray($data);

        return $handler->handle($request);
    }

    private function validateDataArray(array $data): void
    {
        if (!is_array($data)) {
            throw new InvalidArgumentException('Data must be an array');
        }

        if (!array_key_exists('absolute_path', $data)) {
            throw new InvalidArgumentException('`absolute_path` is required');
        }
        
        if (!array_key_exists('backend_path', $data)) {
            throw new InvalidArgumentException('`backend_path` is required');
        }

        $dataWithoutPaths = array_diff_key($data, ['absolute_path' => 1, 'backend_path' => 1]);

        if (empty($dataWithoutPaths)) {
            throw new InvalidArgumentException('Data array cannot be empty');
        }

        $allEmpty = array_reduce(
            array_values($dataWithoutPaths),
            fn($carry, $item) => $carry && empty($item),
            true
        );

        if ($allEmpty) {
            throw new InvalidArgumentException('All data values are empty');
        }
    }
} 