<?php

namespace App\Classes;

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;
use Psr\Log\LoggerInterface;
use App\Traits\RespondsWithJson;
use App\Traits\SensitiveDataFilter;

class Email
{
    use RespondsWithJson;
    use SensitiveDataFilter;

    private PHPMailer $mailer;

    protected array $sensitiveFields = [];

    public function __construct(
        private readonly LoggerInterface $logger,
    ) {
        $this->mailer = new PHPMailer(true);
    }

    private function setupMailer(array $mailConfig): void
    {
        try {
            $this->mailer->isSMTP();
            $this->mailer->Host = $mailConfig['host'];
            $this->mailer->SMTPAuth = true;
            $this->mailer->Username = $mailConfig['username'];
            $this->mailer->Password = $mailConfig['password'];
            $this->mailer->SMTPSecure = $mailConfig['smtp_secure'];
            $this->mailer->Port = $mailConfig['port'];
            $this->mailer->CharSet = 'UTF-8';
        } catch (Exception $e) {
            $this->logger->error("Mailer setup failed: {$e->getMessage()}");
            throw new Exception("Mailer setup failed: {$e->getMessage()}");
        }
    }

    public function send(array $componentConfig, array $data, array $attachments): array
    {
        try {
            // $this->logger->info(print_r($componentConfig['email'], true));
            $this->setupMailer($componentConfig['email']);
            $this->mailer->clearAttachments();

            $from = $componentConfig['email']['to_address'];
            $fromName = $componentConfig['email']['to_name'];

            // Only set reply-to if `email` is provided in data and is valid
            if (!empty($data['email']) && filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
                $replyToName = !empty($data['name']) ? $data['name'] : $componentConfig['email']['to_name'];
                $this->mailer->addReplyTo($data['email'], $replyToName);
            }

            $subject = $data['subject'] ?? "New form submission";

            // Set recipient(s)
            $this->mailer->addAddress($componentConfig['email']['to_address'], $componentConfig['email']['to_name']);  // Primary recipient

            // Remove sensitive data
            $data = $this->removeSensitiveData($data);

            // Create email body from all data fields
            $bodyHtml = "<div style='font-family: Arial, sans-serif;'>\n";
            foreach ($data as $key => $value) {
                // Skip internal fields
                if (in_array($key, ['from', 'from_name', 'subject'])) {
                    continue;
                }

                // Format key for display
                $label = ucwords(str_replace('_', ' ', $key));

                // Handle arrays or objects
                if (is_array($value) || is_object($value)) {
                    $value = json_encode($value);
                }

                $bodyHtml .= sprintf(
                    "<p><strong>%s:</strong> %s</p>\n",
                    htmlspecialchars($label, ENT_QUOTES, 'UTF-8'),
                    nl2br(htmlspecialchars((string)$value, ENT_QUOTES, 'UTF-8'))
                );
            }
            $bodyHtml .= "</div>";

            $this->mailer->setFrom($from, $fromName);
            $this->mailer->isHTML(true);
            $this->mailer->Subject = $subject;
            $this->mailer->Body = $bodyHtml;

            // Set plain text version
            $this->mailer->AltBody = strip_tags(str_replace(
                ['<p>', '</p>', '<br>', '<br/>'],
                ["\n", "\n", "\n", "\n"],
                $bodyHtml
            ));

            // Add uploaded file attachments
            foreach ($attachments as $attachment) {
                // Handle array of attachments
                if (is_array($attachment) && isset($attachment[0])) {
                    foreach ($attachment as $subAttachment) {
                        $this->addSingleAttachment($subAttachment);
                    }
                } else {
                    $this->addSingleAttachment($attachment);
                }
            }

            $this->mailer->send();

            return [
                'success' => true
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    public function process(array $data): array
    {
        return $this->removeSensitiveData($data);
    }

    private function addSingleAttachment($attachment): void
    {
        // Skip if attachment is empty
        if (empty($attachment)) {
            return;
        }

        if (is_array($attachment)) {
            // Skip if required path is missing for array attachments
            if (empty($attachment['path'])) {
                $this->logger->warning("Skipping array attachment: path is empty");
                return;
            }

            $this->logger->info("Adding array attachment with path: {$attachment['path']}");

            $this->mailer->addAttachment(
                $attachment['path'],
                $attachment['name'] ?? basename($attachment['path']),
                'base64',
                $attachment['type'] ?? 'application/octet-stream'
            );
        } elseif ($attachment instanceof \Psr\Http\Message\UploadedFileInterface) {
            // Skip if the upload has an error or is empty
            if ($attachment->getError() !== UPLOAD_ERR_OK || $attachment->getSize() === 0) {
                $this->logger->warning("Skipping uploaded file: has error or zero size");
                return;
            }

            // Create temporary file
            $tmpPath = tempnam(sys_get_temp_dir(), 'email_');
            $stream = $attachment->getStream();
            $stream->rewind();

            // Write uploaded file to temp file
            file_put_contents($tmpPath, $stream->getContents());

            // Add to email
            $this->mailer->addAttachment(
                $tmpPath,
                $attachment->getClientFilename(),
                'base64',
                $attachment->getClientMediaType()
            );

            // Register temp file for cleanup
            register_shutdown_function(function () use ($tmpPath) {
                if (file_exists($tmpPath)) {
                    unlink($tmpPath);
                }
            });
        }
    }
}
