<?php

namespace App\Controllers;

use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;
use App\Classes\WebhookClient;
use InvalidArgumentException;
use Psr\Log\LoggerInterface;
use App\Traits\RespondsWithJson;
use CURLFile;

readonly class WebhookController
{
    use RespondsWithJson;

    public function __construct(
        protected LoggerInterface $logger,
        protected WebhookClient $webhook
    ) {}

    public function create(Request $request, Response $response): Response
    {
        try {
            $componentConfig = requireBackendFile($request, 'config.php');
            $this->validateConfig($componentConfig);

            $data = $request->getParsedBody();
            $files = $request->getUploadedFiles();
            $file = null;

            // Get the first set of files, if any exist
            if (!empty($files) && is_array($files)) {
                $file = reset($files);
            }

            $data = array_merge($data, $file ?? []);

            return $this->jsonResponse(
                $response,
                ['success' => $this->webhook->send($componentConfig['webhook']['url'], $data)]
            );
        } catch (\Exception $e) {
            return $this->errorResponse($response, $e->getMessage(), 500);
        }
    }

    private function validateConfig(array $config): void
    {
        if (empty($config['webhook']['url'])) {
            throw new InvalidArgumentException("No webhook URL configured for component");
        }
    }

    private function getUploadErrorMessage($error)
    {
        return match ($error) {
            UPLOAD_ERR_OK => 'No error',
            UPLOAD_ERR_INI_SIZE => 'File exceeds upload_max_filesize',
            UPLOAD_ERR_FORM_SIZE => 'File exceeds MAX_FILE_SIZE',
            UPLOAD_ERR_PARTIAL => 'File only partially uploaded',
            UPLOAD_ERR_NO_FILE => 'No file was uploaded',
            UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary folder',
            UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
            UPLOAD_ERR_EXTENSION => 'File upload stopped by extension',
            default => 'Unknown upload error'
        };
    }
}
