<?php

namespace App\Classes;

use GuzzleHttp\Client;
use Psr\Log\LoggerInterface;
use InvalidArgumentException;

class WebhookClient
{
    private const MAX_RETRIES = 3;
    private const RETRY_DELAY_MS = 1000;
    private const TIMEOUT_SECONDS = 30;

    private Client $client;

    public function __construct(
        private LoggerInterface $logger
    ) {
        $this->client = new Client([
            'timeout' => self::TIMEOUT_SECONDS,
            'connect_timeout' => 5,
            'http_errors' => true,
            'verify' => true
        ]);
    }

    public function send(string $webhookUrl, array $data): array
    {
        $sanitizedData = $this->sanitizeData($data);

        $response = $this->client->post($webhookUrl, [
            'multipart' => $this->prepareMultipartData($sanitizedData),
            'headers' => [
                'User-Agent' => 'WebhookClient/1.0'
            ]
        ]);

        return json_decode($response->getBody(), true);
    }

    private function prepareMultipartData(array $data): array
    {
        $multipart = [];
        foreach ($data as $key => $value) {
            if ($value instanceof \Psr\Http\Message\UploadedFileInterface) {

                $this->logger->info("Adding file: " . $value->getClientFilename());

                // check for file errors
                if ($value->getError() !== UPLOAD_ERR_OK) {
                    $this->logger->error("File error: " . $value->getError());
                    continue;
                }

                // check for file size
                if (!$value->getSize()) {
                    $this->logger->error("File size error: " . $value->getSize());
                    continue;
                }

                $multipart[] = [
                    'name' => 'file',
                    'filename' => $value->getClientFilename(),
                    'contents' => $value->getStream(),
                    'headers' => ['Content-Type' => $value->getClientMediaType()]
                ];
            } else {
                $multipart[] = [
                    'name' => $key,
                    'contents' => is_array($value) ? json_encode($value) : $value
                ];
            }
        }

        return $multipart;
    }

    private function sanitizeData(array $data): array
    {
        $sensitiveKeys = ['backend_path', 'absolute_path', 'password', 'captcha_token', 'token', 'key', 'secret', 'auth'];

        return array_map(
            fn($value) => is_array($value) ? $this->sanitizeData($value) : $value,
            array_filter(
                $data,
                fn($key) => !in_array(strtolower((string)$key), $sensitiveKeys),
                ARRAY_FILTER_USE_KEY
            )
        );
    }
}
